<?php

/*
 * this class should be used to stores properties and methods shared by the
 * admin and public side of wordpress
 */

class dalt_Shared
{

    //regex
    public $regex_capability = '/^\s*[A-Za-z0-9_]+\s*$/';
    public $list_of_comma_separated_numbers = '/^(\s*(\d+\s*,\s*)+\d+\s*|\s*\d+\s*)$/';
    public $url_regex = '/^https?:\/\/.+$/';
    public $hex_rgb_regex = '/^#(?:[0-9a-fA-F]{3}){1,2}$/';
    public $font_family_regex = '/^([A-Za-z0-9-\'", ]*)$/';
    public $digits_regex = '/^\s*\d+\s*$/';

    protected static $instance = null;

    private $data = array();

    private function __construct()
    {

        //Set plugin textdomain
        load_plugin_textdomain('dalt', false, 'league-table/lang/');

        $this->data['slug'] = 'dalt';
        $this->data['ver'] = '2.17';
        $this->data['dir'] = substr(plugin_dir_path(__FILE__), 0, -7);
        $this->data['url'] = substr(plugin_dir_url(__FILE__), 0, -7);

	    //Here are stored the plugin option with the related default values
	    $this->data['options'] = [

		    //database version -----------------------------------------------------
		    $this->get( 'slug' ) . "_database_version"                                   => "0",
		    $this->get( 'slug' ) . "_import_from_1_x"                                    => "0",

		    //general --------------------------------------------------------------
		    $this->get( 'slug' ) . "_tables_menu_capability"                             => "manage_options",
		    $this->get( 'slug' ) . "_import_menu_capability"                             => "manage_options",
		    $this->get( 'slug' ) . "_export_menu_capability"                             => "manage_options",
		    $this->get( 'slug' ) . "_general_javascript_file_url"                        => $this->get( 'url' ) . 'public/assets/js/general-min.js',
		    $this->get( 'slug' ) . "_general_stylesheet_file_url"                        => $this->get( 'url' ) . 'public/assets/css/general-min.css',
		    $this->get( 'slug' ) . "_tablesorter_library_url"                            => $this->get( 'url' ) . 'public/assets/js/tablesorter/jquery.tablesorter-min.js',
		    $this->get( 'slug' ) . "_load_google_font_1"                                 => "https://fonts.googleapis.com/css2?family=Open+Sans&display=swap",
		    $this->get( 'slug' ) . "_load_google_font_2"                                 => "",
		    $this->get( 'slug' ) . "_max_execution_time"                                 => "300",
		    $this->get( 'slug' ) . "_memory_limit"                                       => "512",
		    $this->get( 'slug' ) . "_limit_shortcode_parsing"                            => "1",
		    $this->get( 'slug' ) . "_verify_single_shortcode"                            => "1",
		    $this->get( 'slug' ) . "_widget_text_shortcode"                              => "0",
		    $this->get( 'slug' ) . "_apply_kses"                                         => "1",
		    $this->get( 'slug' ) . "_kses_allowed_html_tags"                             => "a[href][title], br, em, strong",
		    $this->get( 'slug' ) . "_kses_allowed_protocols"                             => "http, https, ftp, mailto, news, irc, gopher, nntp, feed, telnet",

		    //cell properties
		    $this->get( 'slug' ) . "_enable_text_color_cell_property"                    => "1",
		    $this->get( 'slug' ) . "_enable_background_color_cell_property"              => "1",
		    $this->get( 'slug' ) . "_enable_alignment_cell_property"                     => "1",
		    $this->get( 'slug' ) . "_enable_font_weight_cell_property"                   => "1",
		    $this->get( 'slug' ) . "_enable_font_style_cell_property"                    => "1",
		    $this->get( 'slug' ) . "_enable_link_cell_property"                          => "1",
		    $this->get( 'slug' ) . "_enable_link_color_cell_property"                    => "1",
		    $this->get( 'slug' ) . "_enable_open_link_new_tab_cell_property"             => "1",
		    $this->get( 'slug' ) . "_enable_image_left_cell_property"                    => "1",
		    $this->get( 'slug' ) . "_enable_image_left_link_cell_property"               => "1",
		    $this->get( 'slug' ) . "_enable_image_left_open_link_new_tab_cell_property"  => "1",
		    $this->get( 'slug' ) . "_enable_image_right_cell_property"                   => "1",
		    $this->get( 'slug' ) . "_enable_image_right_link_cell_property"              => "1",
		    $this->get( 'slug' ) . "_enable_image_right_open_link_new_tab_cell_property" => "1",
		    $this->get( 'slug' ) . "_enable_formula_cell_property"                       => "1",
		    $this->get( 'slug' ) . "_enable_formula_data_cell_property"                  => "1",
		    $this->get( 'slug' ) . "_enable_html_content_cell_property"                  => "1",
		    $this->get( 'slug' ) . "_enable_row_slots_cell_property"                     => "0",
		    $this->get( 'slug' ) . "_enable_column_slots_cell_property"                  => "0",

	    ];

    }

    public static function get_instance()
    {

        if (null == self::$instance) {
            self::$instance = new self;
        }

        return self::$instance;

    }

    //retrieve data
    public function get($index)
    {
        return $this->data[$index];
    }

    /*
     * Create a record of the data table filled with the provided data
     *
     * @param $table_id The id of the table
     * @param $row_index The index of the data structure row
     * @param $row_data_json The data of a single data structure row in the json format
     */
    public function data_insert_record($table_id, $row_index, $row_data_json)
    {

        //save in the db table
        global $wpdb;
        $table_name = $wpdb->prefix . $this->get('slug') . "_data";
        $safe_sql = $wpdb->prepare("INSERT INTO $table_name SET
            table_id = %d,
            row_index = %d,
            content = %s",
            $table_id,
            $row_index,
            $row_data_json
        );

        $query_result = $wpdb->query($safe_sql);

    }

    /*
     * Applies stripslashes to all the properties of an object
     *
     * @param Object
     * @return Object
     */
    public function object_stripslashes($obj)
    {

        $property_a = get_object_vars($obj);

        foreach ($property_a as $key => $value) {

            $obj->{$key} = stripslashes($value);

        }

        return $obj;

    }

    /*
     * If the data of the cell exists updates the data based on the provided $cell_info Object.
     * If the data of the cell doesn't exist creates a new record based on the provided $cell_info Object.
     *
     * @param $cell_info
     * @return The result of the query
     */
    public function save_cell($cell_info)
    {

        //Verifies if the cell already exists
        global $wpdb;
        $table_name = $wpdb->prefix . $this->get('slug') . "_cell";
        $safe_sql = $wpdb->prepare("SELECT COUNT(*) FROM $table_name WHERE
        table_id = %d AND row_index = %d AND column_index = %d",
            $cell_info->table_id,
            $cell_info->row_index,
            $cell_info->column_index);
        $cell_count = $wpdb->get_var($safe_sql);

        if ($cell_count > 0) {

            //update the cell data properties
            global $wpdb;
            $table_name = $wpdb->prefix . $this->get('slug') . "_cell";
            $safe_sql = $wpdb->prepare("UPDATE $table_name SET
            html_content = %s,
            text_color = %s,
            background_color = %s,
            font_weight = %s,
            font_style = %s,
            link = %s,
            link_color = %s,
            open_link_new_tab = %d,
            image_left = %s,
            image_left_link = %s,
            image_left_open_link_new_tab = %d,
            image_right = %s,
            image_right_link = %s,
            image_right_open_link_new_tab = %d,
            alignment = %s,
            formula = %s,
            formula_data = %s,
            row_slots = %d,
            column_slots = %d
            WHERE table_id = %d AND row_index = %d AND column_index = %d",
                $cell_info->html_content,
                $cell_info->text_color,
                $cell_info->background_color,
                $cell_info->font_weight,
                $cell_info->font_style,
                $cell_info->link,
                $cell_info->link_color,
                $cell_info->open_link_new_tab,
                $cell_info->image_left,
                $cell_info->image_left_link,
                $cell_info->image_left_open_link_new_tab,
                $cell_info->image_right,
                $cell_info->image_right_link,
                $cell_info->image_right_open_link_new_tab,
                $cell_info->alignment,
                $cell_info->formula,
                $cell_info->formula_data,
	            $cell_info->row_slots,
	            $cell_info->column_slots,
                $cell_info->table_id,
                $cell_info->row_index,
                $cell_info->column_index);
            $result = $wpdb->query($safe_sql);

        } else {

            //if the properties of the cell don't exist create the cell data properties
            global $wpdb;
            $table_name = $wpdb->prefix . $this->get('slug') . "_cell";
            $safe_sql = $wpdb->prepare("INSERT INTO $table_name SET
            html_content = %s,
            text_color = %s,
            background_color = %s,
            font_weight = %s,
            font_style = %s,
            link = %s,
            link_color = %s,
            open_link_new_tab = %d,
            image_left = %s,
            image_left_link = %s,
            image_left_open_link_new_tab = %d,
            image_right = %s,
            image_right_link = %s,
            image_right_open_link_new_tab = %d,
            alignment = %s,
            formula = %s,
            formula_data = %s,
            row_slots = %d,
            column_slots = %d,
            table_id = %d,
            row_index = %d,
            column_index = %d",
                $cell_info->html_content,
                $cell_info->text_color,
                $cell_info->background_color,
                $cell_info->font_weight,
                $cell_info->font_style,
                $cell_info->link,
                $cell_info->link_color,
                $cell_info->open_link_new_tab,
                $cell_info->image_left,
                $cell_info->image_left_link,
                $cell_info->image_left_open_link_new_tab,
                $cell_info->image_right,
                $cell_info->image_right_link,
                $cell_info->image_right_open_link_new_tab,
                $cell_info->alignment,
                $cell_info->formula,
                $cell_info->formula_data,
                $cell_info->row_slots,
                $cell_info->column_slots,
                $cell_info->table_id,
                $cell_info->row_index,
                $cell_info->column_index);
            $result = $wpdb->query($safe_sql);

        }

        return $result;

    }

    /*
     * Returns the total number of non-temporary tables
     *
     * @return int The number of non-temporary tables
     */
    public function get_number_of_tables()
    {

        global $wpdb;
        $table_name = $wpdb->prefix . $this->get('slug') . "_table";
        $number_of_tables = $wpdb->get_var("SELECT COUNT(*) FROM $table_name WHERE temporary = 0");

        return $number_of_tables;

    }

    /*
     * Applies wp_kses() with the allowed tags and parameters defined in the plugin options.
     *
     * @param String The HTML that should be filtered
     * @return String The HTML filter by wp_kses() (if "Apply kses" is enabled) or the unfilter HTML (if "Apply kses" is
     * not enabled).
     */
    public function apply_custom_kses($string)
    {

        if (get_option($this->get('slug') . '_apply_kses') == 1) {

            /*
             * The final $allowed_html should have this format, for more info see
             * https://codex.wordpress.org/Function_Reference/wp_kses
             *
             * $allowed_html = array(
             *     'a' => array(
             *         'href' => array(),
             *         'title' => array()
             *     ),
             *     'br' => array(),
             *     'em' => array(),
             *     'strong' => array(),
             * );
             */

            $allowed_html = array();
            $kses_allowed_html_tags = get_option($this->get('slug') . '_kses_allowed_html_tags');
            $list_of_tags_a = explode(',', $kses_allowed_html_tags);

            foreach ($list_of_tags_a as $tag_with_attributes) {

                //store the allowed attributes of this tag in an array with a regex
                preg_match_all('/\[([a-z-]*)\]/', $tag_with_attributes, $matches);
                $allowed_attributes_a = $matches[1];

                //remove the allowed attributes from the string and store the allowed tag in a variable with a regex
                $allowed_tag = trim(preg_replace('(\[[a-z-]*\])', '', $tag_with_attributes));

                //create an array with the proper form required by the wp_kses() function
                $attributes = array();
                foreach ($allowed_attributes_a as $allowed_attribute) {
                    $attributes[$allowed_attribute] = array();
                }

                //add the tag only if it's validated with a regex
                if (preg_match('/^[a-z]*$/', $allowed_tag)) {
                    $allowed_html[$allowed_tag] = $attributes;
                }

            }

            /*
             * Set the allowed protocols.
             *
             * The final $allowed_protocols should have this format, for more info see
             * https://codex.wordpress.org/Function_Reference/wp_kses
             *
             * $allowed_protocols = array('http', 'https', 'ftp');
             */

            //get the value in the option
            $allowed_protocols = get_option($this->get('slug') . '_kses_allowed_protocols');

            //remove all the spaces
            $allowed_protocols = str_replace(' ', '', $allowed_protocols);

            //convert to an array
            $allowed_protocols = explode(',', $allowed_protocols);

            //Use the custom $allowed_html and $allowed_protocols to apply wp_kses()
            return wp_kses($string, $allowed_html, $allowed_protocols);

        } else {

            return $string;

        }

    }

    /*
     * Get the number of columns available in the table.
     *
     * @param int The id of the table
     * @param bool A boolean value which indicated if the position column, generated in the front-end via JavaScript,
     * should be considered in the calculation of the number of columns.
     * @return int The number of columns available in the table.
     */
    function get_number_of_columns($table_id, $consider_position_column){

	    global $wpdb;
	    $number_of_columns = 0;

	    /*
	     * Sum the number of columns found in the serialized field "content" of the data table to $number_of_columns.
		 */
	    $table_name = $wpdb->prefix . $this->get('slug') . "_data";
	    $safe_sql = $wpdb->prepare("SELECT * FROM $table_name WHERE table_id = %d ORDER BY row_index DESC", $table_id);
	    $data_a = $wpdb->get_results($safe_sql);

	    $row_data = json_decode($data_a[0]->content, true);
	    $number_of_columns = $number_of_columns + count($row_data);

	    /*
	     * Add 1 to $number_of_columns if:
	     *
	     * - The position column should be considered
	     * - If the position column is enabled in this table
	     */
	    $table_name = $wpdb->prefix . $this->get('slug') . "_table";
	    $safe_sql = $wpdb->prepare("SELECT * FROM $table_name WHERE id = %d", $table_id);
	    $table_obj = $wpdb->get_row($safe_sql);

	    if(intval($consider_position_column, 10) ===1 and intval($table_obj->show_position, 10) === 1){
		    $number_of_columns = $number_of_columns + 1;
	    }

		return $number_of_columns;

    }

	/**
	 * Returns the value of the "rows" field of the "table" db table
	 *
	 * @param $table_id The table id
	 * @return Int the value of the "rows" field
	 */
	function get_rows_field($table_id){

		global $wpdb;
		$table_name = $wpdb->prefix . $this->get('slug') . "_table";
		$safe_sql = $wpdb->prepare("SELECT `rows` FROM $table_name WHERE id = %d", $table_id);
		$table_obj = $wpdb->get_row($safe_sql);

		return intval($table_obj->rows, 10);

	}

	/**
	 * Returns the value of the "columns" field of the "table" db table
	 *
	 * @param $table_id The table id
	 * @return Int the value of the "columns" field
	 */
    function get_columns_field($table_id){

    	global $wpdb;
	    $table_name = $wpdb->prefix . $this->get('slug') . "_table";
	    $safe_sql = $wpdb->prepare("SELECT columns FROM $table_name WHERE id = %d", $table_id);
	    $table_obj = $wpdb->get_row($safe_sql);

	    return intval($table_obj->columns, 10);

    }

	/*
	 * Empty objects are replaced with empty strings.
	 * This prevent to generate notices with the methods of the wpdb class.
	 *
	 * @param $data An array which includes empty objects that should be converted to empty strings
	 * @return string An array where the empty objects have been replaced with empty strings
	 */
	public function replace_empty_objects_with_empty_strings($data){

		foreach($data as $key => $value){
			if(gettype($value) === 'object'){

				/**
				 * Verify if the $value object is empty by typecasting it into an array. In case it's empty replace its
				 * value with an empty string.
				 *
				 * Ref: https://stackoverflow.com/questions/9412126/how-to-check-that-an-object-is-empty-in-php
				 */
				$arr = (array)$value;
				if (empty($arr)) {
					$data[$key] = '';
				}

			}
		}

		return $data;

	}

	/**
	 * Check the AJAX referer.
	 */
	public function check_ajax_referer(){

		//check the referer
		if (!check_ajax_referer('dalt', 'security', false)) {
			echo "Invalid AJAX Request";
			die();
		}

	}

	/**
	 * Check the tables menu capability.
	 */
	public function check_tables_menu_capability(){

		//check the capability
		if (!current_user_can(get_option($this->get('slug') . "_tables_menu_capability"))) {
			esc_html_e('Invalid Capability', 'dalt');
			die();
		}

	}

	/**
	 * Set the maximum execution time defined in the "Max Execution Time" option.
	 */
	public function set_max_execution_time(){

		ini_set('max_execution_time', intval(get_option($this->get('slug') . "_max_execution_time"), 10));

	}

	/**
	 * Set the memory limit in megabytes defined in the "Memory limit" option.
	 */
	public function set_memory_limit(){

		ini_set('memory_limit', intval(get_option($this->get('slug') . "_memory_limit"), 10) . 'M');

	}

	/**
	 * Sanitize the data of the table provided as an escaped json string.
	 *
	 * @param $table_data
	 *
	 * @return array|bool
	 */
	public function sanitize_table_data($table_data){

		//Unescape and decode the table data provided in json format
		$table_data = json_decode(stripslashes($table_data));

		//Verify if data property of the returned object is an array
		if(!isset($table_data->data) or !is_array($table_data->data)){
			return false;
		}

		//Save the two dimensional array that include the table data in the $table_data variable
		$table_data = $table_data->data;

		foreach ($table_data as $row_index => $row_data) {

			//Verify if the table row data are provided as an array
			if(!is_array($row_data)){
				return false;
			}

			//sanitize all the cells data in the $row_data array
			$table_data[$row_index] = array_map('sanitize_text_field', $row_data);

		}

		return $table_data;

	}

	/**
	 * Sanitize the cell properties of the provided group of cells.
	 *
	 * @param $cell_properties
	 *
	 * @return array|mixed|object
	 */
	public function sanitize_cell_properties_multiple($cell_properties){

		foreach($cell_properties as $key_row => $cell_properties_row) {

			foreach ( $cell_properties_row as $key_column => $cell_properties_cell ) {

				if($cell_properties_cell !== false){

					$cell_properties[$key_row][$key_column] = $this->sanitize_cell_properties($cell_properties_cell);

				}

			}

		}

		return $cell_properties;

	}

	/**
	 * Sanitize the cell properties of a single cell.
	 *
	 * @param $cell_properties
	 *
	 * @return mixed
	 */
	function sanitize_cell_properties($cell_properties){

		$cell_properties['table_id'] = intval($cell_properties['table_id'], 10);
		$cell_properties['row_index'] = intval($cell_properties['row_index'], 10);
		$cell_properties['column_index'] = intval($cell_properties['column_index'], 10);
		$cell_properties['html_content'] = $this->apply_custom_kses($cell_properties['html_content']);
		$cell_properties['text_color'] = sanitize_hex_color($cell_properties['text_color']);
		$cell_properties['background_color'] = sanitize_hex_color($cell_properties['background_color']);
		$cell_properties['font_weight'] = sanitize_text_field($cell_properties['font_weight']);
		$cell_properties['font_style'] = sanitize_key($cell_properties['font_style']);
		$cell_properties['link'] = esc_url_raw($cell_properties['link']);
		$cell_properties['link_color'] = sanitize_hex_color($cell_properties['link_color']);
		$cell_properties['open_link_new_tab'] = intval($cell_properties['open_link_new_tab'], 10);
		$cell_properties['image_left'] = esc_url_raw($cell_properties['image_left']);
		$cell_properties['image_left_link'] = esc_url_raw($cell_properties['image_left_link']);
		$cell_properties['image_left_open_link_new_tab'] = intval($cell_properties['image_left_open_link_new_tab'], 10);
		$cell_properties['image_right'] = esc_url_raw($cell_properties['image_right']);
		$cell_properties['image_right_link'] = esc_url_raw($cell_properties['image_right_link']);
		$cell_properties['image_right_open_link_new_tab'] = intval($cell_properties['image_right_open_link_new_tab'], 10);
		$cell_properties['alignment'] = sanitize_key($cell_properties['alignment']);
		$cell_properties['formula'] = sanitize_key($cell_properties['formula']);
		$cell_properties['formula_data'] = sanitize_text_field($cell_properties['formula_data']);
		$cell_properties['row_slots'] = intval($cell_properties['row_slots'], 10);
		$cell_properties['column_slots'] = intval($cell_properties['column_slots'], 10);

		return $cell_properties;

	}

	/**
	 * Sanitize the data of cell selection.
	 *
	 * @param $cell_selection
	 *
	 * @return object|bool
	 */
	public function sanitize_cell_selection($cell_selection){

		//Unescape and decode the table data provided in json format
		$cell_selection = json_decode(stripslashes($cell_selection));

		//Verify if data property of the returned object is an array
		if(!isset($cell_selection->start->row) or
		   !isset($cell_selection->start->col) or
		   !isset($cell_selection->end->row) or
		   !isset($cell_selection->end->col)){
			return false;
		}

		$cell_selection->start->row = intval($cell_selection->start->row, 10);
		$cell_selection->start->col = intval($cell_selection->start->col, 10);
		$cell_selection->end->row = intval($cell_selection->end->row, 10);
		$cell_selection->end->col = intval($cell_selection->end->col, 10);

		return $cell_selection;

	}

	/**
	 * Adds global javascript parameters before the script with the specified handle.
	 *
	 * @param $handle
	 */
	public function add_global_javascript_parameters($handle){

		//Store the JavaScript parameters in the window.DALT_PARAMETERS object
		$initialization_script = 'window.DALT_PARAMETERS = {';
		$initialization_script .= 'ajax_url: "' . admin_url('admin-ajax.php') . '",';
		$initialization_script .= 'nonce: "' . wp_create_nonce("dalt") . '",';
		$initialization_script .= 'admin_url: "' . get_admin_url() . '"';
		$initialization_script .= '};';
		wp_add_inline_script( $this->get('slug') . $handle, $initialization_script, 'before' );

	}

}